// AI Service for Content Generation
class AIService {
    static async generateContent(topic, settings, onProgress) {
        const apiKey = await this.getUserAPIKey();
        if (!apiKey) {
            throw new Error('Access Code not configured. Please save your Access Code first.');
        }

        const url = `https://generativelanguage.googleapis.com/v1beta/models/gemini-2.0-flash:generateContent?key=${apiKey}`;
        
        // Enhanced prompt with proper integration of custom prompts and content type
        const enhancedPrompt = await this.enhancePrompt(topic, settings);
        
        const requestBody = {
            contents: [{
                parts: [{
                    text: enhancedPrompt
                }]
            }],
            generationConfig: {
                temperature: settings.creativity / 100,
                maxOutputTokens: Math.min(settings.wordCount * 2, 8192),
                topP: 0.95,
                topK: 40
            }
        };

        try {
            const response = await fetch(url, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(requestBody)
            });

            if (!response.ok) {
                const errorData = await response.json();
                throw new Error(`API request failed: ${response.status} ${response.statusText} - ${errorData.error?.message || 'Unknown error'}`);
            }

            const data = await response.json();
            
            if (!data.candidates || !data.candidates[0] || !data.candidates[0].content) {
                throw new Error('Invalid response from AI API: No content generated');
            }

            const content = data.candidates[0].content.parts[0].text;
            
            // Simulate streaming effect
            if (onProgress) {
                await this.simulateStreaming(content, onProgress);
            }
            
            // Save content to database
            await this.saveContentToDB(settings.currentTopic, content, settings);
            
            return content;
            
        } catch (error) {
            console.error('AI API Error:', error);
            throw new Error(`Failed to generate content: ${error.message}`);
        }
    }

    static async getUserAPIKey() {
        // Try to get user's API key first
        try {
            const response = await fetch('ai/get_user_keys.php');
            const data = await response.json();
            if (data.success && data.ai_api_key) {
                return data.ai_api_key;
            }
        } catch (error) {
            console.warn('Failed to get user API key:', error);
        }
        
        // Fallback to default key
        return 'AIzaSyDbjf0QmBkJ1Mc0gMt0Wwp2B68nygD6zKs';
    }

    static async enhancePrompt(topic, settings) {
        const personaPrompt = await this.getPersonaPrompt(settings.aiPersona);
        const contentTypeGuidelines = this.getContentTypeGuidelines(settings.contentType);
        
        // Integrate custom prompt if provided
        let customInstructions = '';
        if (topic.prompt && topic.prompt.trim()) {
            customInstructions = `\n\nADDITIONAL USER INSTRUCTIONS:\n${topic.prompt}\n\nIMPORTANT: You MUST integrate these additional instructions naturally into the ${settings.contentType} about "${topic.title}". Do not ignore or separate these instructions.`;
        }

        return `${personaPrompt}

PRIMARY TASK: Create a ${settings.contentType} about "${topic.title}"

${contentTypeGuidelines}

TONE REQUIREMENT: Write in a ${settings.tone} tone.

WORD COUNT TARGET: Approximately ${settings.wordCount} words.

${customInstructions}

CRITICAL REQUIREMENTS:
1. Stay 100% focused on the topic: "${topic.title}"
2. Follow the ${settings.contentType} format and structure exactly
3. Integrate any additional instructions naturally into the content
4. Maintain a ${settings.tone} tone throughout
5. Include practical, actionable insights
6. Use real-world examples and evidence
7. Structure content for optimal readability using paragraphs, H1, H2, H3, H4, bold, italics, underline, hyperlinks, etc when necessary
8. Ensure content flows naturally and logically

CONTENT STRUCTURE:
- Start with an engaging introduction that hooks the reader
- Use clear headings and subheadings
- Include specific examples and data where relevant
- End with a strong conclusion that summarizes key points
- Provide actionable takeaways

Remember: You are creating a ${settings.contentType}, so follow the appropriate format and depth for this content type while staying completely focused on "${topic.title}".`;
    }

    static getContentTypeGuidelines(contentType) {
        const guidelines = {
            'article': 'Create a comprehensive, well-researched article with an introduction, body with multiple sections, and conclusion. Include subheadings, examples, and practical insights.',
            'listicle': 'Create a numbered list format with clear, scannable points. Each point should be substantial and valuable. Include an introduction and conclusion.',
            'news': 'Write in a journalistic style with the most important information first. Follow the inverted pyramid structure. Include who, what, when, where, why, and how.',
            'product': 'Focus on features, benefits, and user experience. Include specifications, use cases, and comparisons if relevant. Be persuasive but factual.',
            'social': 'Create engaging, concise content optimized for social media. Use emojis sparingly and include calls to action.',
            'email': 'Write persuasive email copy with a clear subject line, engaging opening, valuable content, and strong call to action.',
            'press': 'Follow press release format: headline, dateline, body, boilerplate. Use formal, professional language.',
            'case': 'Present a problem, solution, implementation, and results. Use specific metrics and testimonials if possible.',
            'whitepaper': 'Provide in-depth, research-backed analysis. Use formal academic style with citations and data.',
            'guide': 'Create step-by-step instructions with clear explanations. Assume the reader is a beginner.',
            'tutorial': 'Provide detailed, sequential instructions with examples. Include troubleshooting tips.',
            'faq': 'Use question-and-answer format. Group related questions together and provide comprehensive answers.',
            'landing': 'Write persuasive copy focused on conversions. Use headlines, subheadlines, benefits, and strong CTAs.',
            'about': 'Tell a compelling story about the subject. Focus on mission, values, and unique qualities.',
            'script': 'Write in conversational format with speaker directions. Include natural dialogue and pacing.',
            'podcast': 'Create an engaging outline with introduction, main segments, and conclusion. Include talking points and transitions.'
        };

        return guidelines[contentType] || guidelines['article'];
    }

    static async getPersonaPrompt(personaKey) {
        try {
            const response = await fetch(`ai/get_persona.php?persona=${personaKey}`);
            const data = await response.json();
            if (data.success) {
                return data.persona.prompt_template;
            }
        } catch (error) {
            console.warn('Failed to get persona prompt:', error);
        }
        
        // Fallback persona prompt
        return 'You are an expert content creator with deep industry knowledge and practical experience. Your writing is clear, engaging, and valuable to readers.';
    }

    static async saveContentToDB(topic, content, settings) {
        try {
            const formData = new FormData();
            formData.append('topic', topic);
            formData.append('content', content);
            formData.append('word_count', content.split(/\s+/).length);
            formData.append('content_type', settings.contentType);
            formData.append('tone', settings.tone);
            formData.append('ai_persona', settings.aiPersona);

            await fetch('ai/save_content.php', {
                method: 'POST',
                body: formData
            });
        } catch (error) {
            console.warn('Failed to save content to database:', error);
        }
    }

    static async simulateStreaming(content, onProgress) {
        const words = content.split(' ');
        let currentContent = '';
        
        for (let i = 0; i < words.length; i++) {
            currentContent += words[i] + ' ';
            
            // Update progress
            if (onProgress) {
                onProgress(words[i] + ' ', currentContent);
            }
            
            // Random delay to simulate typing
            const delay = Math.random() * 50 + 20;
            await new Promise(resolve => setTimeout(resolve, delay));
            
            // Batch updates for better performance
            if (i % 10 === 0) {
                await new Promise(resolve => setTimeout(resolve, 0));
            }
        }
    }

    static async validateAPIKey(key) {
        if (!key) {
            throw new Error('API key is required');
        }
        
        if (!key.startsWith('AIzaSy')) {
            throw new Error('Invalid API key format. Must start with AIzaSy');
        }
        
        try {
            const url = `https://generativelanguage.googleapis.com/v1beta/models/gemini-2.0-flash:generateContent?key=${key}`;
            
            const response = await fetch(url, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    contents: [{
                        parts: [{
                            text: "Hello, please respond with 'OK' if you can read this."
                        }]
                    }],
                    generationConfig: {
                        maxOutputTokens: 10
                    }
                })
            });

            if (!response.ok) {
                throw new Error(`API test failed: ${response.status}`);
            }

            const data = await response.json();
            return data.candidates && data.candidates[0];
            
        } catch (error) {
            throw new Error(`Connection test failed: ${error.message}`);
        }
    }
}